<?php
namespace Maxhost_Elementor_Extensions;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

final class Plugin {

	/**
	 * Minimum Elementor Version
	 *
	 * @since 1.0.0
	 * @var string Minimum Elementor version required to run the addon.
	 */
	const MINIMUM_ELEMENTOR_VERSION = '3.5.0';

	/**
	 * Minimum PHP Version
	 *
	 * @since 1.0.0
	 * @var string Minimum PHP version required to run the addon.
	 */
	const MINIMUM_PHP_VERSION = '7.1';
	
	/**
	 * Instance
	 *
	 * @since 1.0.0
	 *
	 * @access private
	 * @static
	 *
	 * @var Plugin The single instance of the class.
	 */
	private static $_instance = null;

	/**
	 * Instance
	 *
	 * Ensures only one instance of the class is loaded or can be loaded.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 * @static
	 *
	 * @return Plugin An instance of the class.
	 */
	public static function instance() {

		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;

	}
	
	/**
	 * Constructor
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function __construct() {

		if ( $this->is_compatible() ) {
			add_action( 'elementor/init', [ $this, 'init' ] );
		}
	}
	
	/**
	 * Compatibility Checks
	 *
	 * Checks whether the site meets the addon requirement.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function is_compatible() {

		// Check if Elementor installed and activated
		if ( ! did_action( 'elementor/loaded' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_missing_main_plugin' ] );
			return false;
		}

		// Check for required Elementor version
		if ( ! version_compare( ELEMENTOR_VERSION, self::MINIMUM_ELEMENTOR_VERSION, '>=' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_elementor_version' ] );
			return false;
		}

		// Check for required PHP version
		if ( version_compare( PHP_VERSION, self::MINIMUM_PHP_VERSION, '<' ) ) {
			add_action( 'admin_notices', [ $this, 'admin_notice_minimum_php_version' ] );
			return false;
		}

		return true;

	}
	
	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have Elementor installed or activated.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function admin_notice_missing_main_plugin() {

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor */
			esc_html__( '"%1$s" requires "%2$s" to be installed and activated.', 'maxhost' ),
			'<strong>' . esc_html__( 'Elemontor Widgets of Maxhost', 'maxhost' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'maxhost' ) . '</strong>'
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required Elementor version.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function admin_notice_minimum_elementor_version() {

		$message = sprintf(
			/* translators: 1: Plugin name 2: Elementor 3: Required Elementor version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'maxhost' ),
			'<strong>' . esc_html__( 'Elemontor Widgets of Maxhost', 'maxhost' ) . '</strong>',
			'<strong>' . esc_html__( 'Elementor', 'maxhost' ) . '</strong>',
			 self::MINIMUM_ELEMENTOR_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Admin notice
	 *
	 * Warning when the site doesn't have a minimum required PHP version.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function admin_notice_minimum_php_version() {

		$message = sprintf(
			/* translators: 1: Plugin name 2: PHP 3: Required PHP version */
			esc_html__( '"%1$s" requires "%2$s" version %3$s or greater.', 'maxhost' ),
			'<strong>' . esc_html__( 'Elemontor Widgets of Maxhost', 'maxhost' ) . '</strong>',
			'<strong>' . esc_html__( 'PHP', 'maxhost' ) . '</strong>',
			 self::MINIMUM_PHP_VERSION
		);

		printf( '<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message );

	}

	/**
	 * Initialize the plugin
	 *
	 * Load the plugin only after Elementor (and other plugins) are loaded.
	 * Checks for basic plugin requirements, if one check fail don't continue,
	 * if all check have passed load the files required to run the plugin.
	 *
	 * Fired by `plugins_loaded` action hook.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init() {

			// Add Plugin actions
			add_action( 'elementor/elements/categories_registered', array( $this, 'add_widget_categories' ) );
			add_action( 'elementor/widgets/register', array( $this, 'init_widgets' ) );
			add_action( 'elementor/preview/enqueue_styles', array( $this, 'custom_styles' ) );
	}

	
	/**
     * Element Category Register
     */
    public function add_widget_categories( $elements_manager ) {
        $elements_manager->add_category(
            'maxhost-elements',
            [
                'title' => esc_html__( 'Maxhost Elements', 'maxhost' ),
                'icon' => 'fa fa-plug',
            ]
        );
    }
	
	/**
	 * Init Widgets
	 *
	 * Include widgets files and register them
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 */
	public function init_widgets($widgets_manager) {

		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/accor_01.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/accor_02.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/accor_03.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/accor_04.php';
		
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/banner_01.php';
		
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/blog_posts_01.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/blog_posts_02.php';
		
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/bullet_points.php';
		
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/call_to_action.php';
		
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/clients_block.php';
		
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/domain_package.php';
		
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/domain_register.php';
		
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/domain_search.php';
		
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/features_block_01.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/features_block_05.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/features_block_06.php';
		
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/features_table.php';
		
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/features_with_icons_01.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/features_with_icons_02.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/features_with_icons_03.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/features_with_icons_04.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/features_with_icons_05.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/features_with_icons_06.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/features_with_icons_07.php';
		
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/horizontal_tabs_01.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/horizontal_tabs_02.php';
		
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/mega_menu_01.php';
		
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/offers_carousel.php';
		
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/page_header.php';
		
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/pricing_filter.php';
		
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/pricing_table_01.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/pricing_table_02.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/pricing_table_03.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/pricing_table_04.php';
		
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/team.php';
		
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/testimonial_01.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/testimonial_02.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/testimonial_03.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/testimonial_04.php';
		
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/title_01.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/title_02.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/title_03.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/title_04.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/title_05.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/title_06.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/title_07.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/title_08.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/title_09.php';
		require_once MAXHOST_PLUGIN_PATH . '/modules/inc/elementor/elements/top_heading_section.php';

		// Register widget
		$widgets_manager->register( new \Maxhost_Elementor_Accor_01() );
		$widgets_manager->register( new \Maxhost_Elementor_Accor_02() );
		$widgets_manager->register( new \Maxhost_Elementor_Accor_03() );
		$widgets_manager->register( new \Maxhost_Elementor_Accor_04() );
		
		$widgets_manager->register( new \Maxhost_Elementor_Banner_01() );
		
		$widgets_manager->register( new \Maxhost_Elementor_Blog_Posts_01() );
		$widgets_manager->register( new \Maxhost_Elementor_Blog_Posts_02() );
		
		$widgets_manager->register( new \Maxhost_Elementor_Bullet_Points() );
		
		$widgets_manager->register( new \Maxhost_Elementor_Call_To_Action() );
		
		$widgets_manager->register( new \Maxhost_Elementor_Clients_Block() );
		
		$widgets_manager->register( new \Maxhost_Elementor_Domain_Package() );
		
		$widgets_manager->register( new \Maxhost_Elementor_Domains_Register() );
		
		$widgets_manager->register( new \Maxhost_Elementor_Domain_Search() );
		
		$widgets_manager->register( new \Maxhost_Elementor_Features_Block_01() );
		$widgets_manager->register( new \Maxhost_Elementor_Features_Block_05() );
		$widgets_manager->register( new \Maxhost_Elementor_Features_Block_06() );
		
		$widgets_manager->register( new \Maxhost_Elementor_Features_Table() );
		
		$widgets_manager->register( new \Maxhost_Elementor_Iconic_Features_01() );
		$widgets_manager->register( new \Maxhost_Elementor_Iconic_Features_02() );
		$widgets_manager->register( new \Maxhost_Elementor_Iconic_Features_03() );
		$widgets_manager->register( new \Maxhost_Elementor_Iconic_Features_04() );
		$widgets_manager->register( new \Maxhost_Elementor_Iconic_Features_05() );
		$widgets_manager->register( new \Maxhost_Elementor_Iconic_Features_06() );
		$widgets_manager->register( new \Maxhost_Elementor_Iconic_Features_07() );
		
		$widgets_manager->register( new \Maxhost_Elementor_Htab_01() );
		$widgets_manager->register( new \Maxhost_Elementor_Htab_02() );
		
		$widgets_manager->register( new \Maxhost_Elementor_Mega_Menu_01() );
		
		$widgets_manager->register( new \Maxhost_Elementor_Offers_Carousel() );
		
		$widgets_manager->register( new \Maxhost_Elementor_Page_Header() );
		
		$widgets_manager->register( new \Maxhost_Elementor_Pricing_Filter() );
		
		$widgets_manager->register( new \Maxhost_Elementor_Pricing_Table_01() );
		$widgets_manager->register( new \Maxhost_Elementor_Pricing_Table_02() );
		$widgets_manager->register( new \Maxhost_Elementor_Pricing_Table_03() );
		$widgets_manager->register( new \Maxhost_Elementor_Pricing_Table_04() );
		
		$widgets_manager->register( new \Maxhost_Elementor_Team() );
		
		$widgets_manager->register( new \Maxhost_Elementor_Testimoni_01() );
		$widgets_manager->register( new \Maxhost_Elementor_Testimoni_02() );
		$widgets_manager->register( new \Maxhost_Elementor_Testimoni_03() );
		$widgets_manager->register( new \Maxhost_Elementor_Testimoni_04() );
		
		$widgets_manager->register( new \Maxhost_Elementor_Title_01() );
		$widgets_manager->register( new \Maxhost_Elementor_Title_02() );
		$widgets_manager->register( new \Maxhost_Elementor_Title_03() );
		$widgets_manager->register( new \Maxhost_Elementor_Title_04() );
		$widgets_manager->register( new \Maxhost_Elementor_Title_05() );
		$widgets_manager->register( new \Maxhost_Elementor_Title_06() );
		$widgets_manager->register( new \Maxhost_Elementor_Title_07() );
		$widgets_manager->register( new \Maxhost_Elementor_Title_08() );
		$widgets_manager->register( new \Maxhost_Elementor_Title_09() );
		$widgets_manager->register( new \Maxhost_Elementor_Top_Heading_Section() );
	}
	
	public function custom_styles() { ?>
		<style id="maxhost_elementor_admin_custom_style">
		.elementor-editor-active div#preloader {display: none !important;}
		</style>
		<?php
    }

}